function [data] = fill_stat(varargin)
%This function helps the user calculate the fill rate statistic. 
%Synatax:   [data] = fill_stat('zfilter',1); 
%Input:     User will be prompted to open datasets.  Lists of
%           vertices and a channel used to calculate volumn.
%           z = the scale factor for z.  Most likely x and y are 1 and 1,
%               thus, z if 100nm will be 1, but 70nm will be .7.  
%               Default = 1
%           vert = open a vertices list to calculate density.  Default = 0
%               (off).
%           border = calculate the border volume or not.  Default  = 0;
%           zfilter = filter the objects based on whether then are in
%               multiple slices or not.  Default = 0;
%Output:    data = the output data structure. Fields:
%               stat = the fill stat.
%               filename = the filenames of the verticies lists, or dapi
%                   filename.
%               pathnames = the path for those files.
%               fill_vox = the fillable number of voxels in the image
%               nucli_stk = the black and white image stack used to
%                       calculate the pixels not being used within the
%                       image, name after nuclei.
%               border_stk = the BW image stack used to calculate the
%                       boarder of the image.
%               obound = the number of pixels on the boundary
%               total_vox = the total number of voxels in the image

%parse input.
[z,vert,border,zfilter] = parse(varargin);

%prompt user to open up vertices lists.
if vert
    prompt_box('title','Open Vertex Lists','prompt1','Select the channels you want analyzed.','position','center');
    pause(0.25);
    vertices = {};      %initialize vertex array
    %get files. Note: currently the filterindex is the filter selected for the file selection.
    [filenames,pathnames,filterindex] = uigetfile2b({'*.csv','Text files (*.csv)';'*.xlsx','Excel 2007 files (*.xlsx)';...
        '*.xls','Excel files (*.xls)';'*.mat','Mat files (*.mat)';'*.*','All Files';},...
        'Open Stack','Multiselect','on');
    setappdata(0,'UseNativeSystemDialogs',true)
    if ~iscell(filenames)     %filename must be a cell array
        filename_tmp{1,1} = filenames;
        filenames = filename_tmp;
        clear filename_tmp
    end

    %Now open the data sets according to the filter used.
    switch filterindex
        case {2,3}              %excel file
            for i = 1:size(filenames,2)
                verti_tmp{i} = single(xlsread([pathnames,filenames{1,i}]));
            end
            vertices = cat(2,vertices,verti_tmp);       %join all the opened data together
        case {1,5}                  %text file - Default coma delimited
            for i = 1:size(filenames,2)
                %verti_tmp{i} = single(dlmread([pathname_tmp,filename_tmp{1,i}],','));
                verti_tmp{i} = single(dlmread([pathnames,filenames{1,i}],',',1,0));
            end
            vertices = cat(2,vertices,verti_tmp);       %join all the opened data together
        otherwise               %MAT FILES      - Note Note supported right now.
            for i = 1:size(filenames,2)
                load([pathnames,filenames{1,i}]);
            end
    end
end

if border   %calculate border volume if desired
    %now open up the Nuclei subtraction image channel
    prompt_box('title','Open Border Reference Stack','prompt1','Select an image stack from the dataset(8bit)',...
        'prompt2','I highly recommend any Green/488 Channel','position','center');
    pause(0.25);
    [stk] = stack_gui;  %use cell output mode.
    stk = imnorm(stk);  %normalize it for better performance
    stk = im2uint8(stk);   %convert to 8bit
    
    %pick the image to use for display.  We want the one with the least
    %contrast
    for k = 1:size(stk,3)   %step through the stack
        img_std(k) = mean(std(single(stk(:,:,k))));     %grab the standard deviation of the image
        img_int(k) = sum(sum(single(stk(:,:,k))));      %grab the total intensity of the image
    end
    img_std = img_std/mean(img_std);    %normalize
    img_int = img_int/mean(img_int);    %ditto
    img_score = img_std+img_int;    %create an image score
    img_ptr(1) = find(img_score==min(img_score));  %We want the lowest score
    img_ptr(2) = find(img_score==max(img_score));  %and the highest score
    
    %find the boundary pixels
    prompt_box('title','Find the border first','prompt1','Lets get interactive... ',...
        'prompt2','default Threshold = 0.00001','position','center');
    pause(0.25);
    [thresh] = play_thresh(stk,img_ptr,0.00001,zfilter);     %use the middle image as the representative image.
    stk_bw2 = false(size(stk));      %preallocate
    for i = 1:size(stk,3)     %make the stk Black & White
        stk_bw2(:,:,i) = im2bw(stk(:,:,i),thresh);        %threshold should work fine at .1, change if you want
        stk_bw2(:,:,i) = imfill(stk_bw2(:,:,i),'holes');    %fill in the holes, 2d fill is actually better, because 3d has holes in the z.
    end
    obound = size(stk_bw2(stk_bw2==0),1);    %get the number of pixels outside of the image area due to rotation and warping.
    data(1).border_thresh = thresh;         %FYI store the threshold
    clear thresh stk
else    %no boarder volume calculated place in place holder volumes
    stk_bw2 = 0;
    obound = 0;    %get the number of pixels outside of the image area due to rotation and warping.
    data(1).border_thresh = 0;         %FYI store the threshold
end

%now open up the Nuclei subtraction image channel
prompt_box('title','Open Nuclei Reference Stack','prompt1','Select an image stack from the dataset(8bit)',...
    'prompt2','I highly recommend DAPI','position','center');
pause(0.25);
[stk,range,filename,pathname] = stack_gui;  %use cell output mode.
se = strel('ball',3,3);     %create the structuring element for morphological operations
stk = imnorm(stk);  %normalize it for better performance
stk = imdilate(stk,se);   %dilate the DAPI channel a little bit
stk = im2uint8(stk);   %convert to 8bit

%pick the image to use for display.  We want the one with the least
%contrast
for k = 1:size(stk,3)   %step through the stack
    img_std(k) = mean(std(single(stk(:,:,k))));     %grab the standard deviation of the image
    img_int(k) = sum(sum(single(stk(:,:,k))));      %grab the total intensity of the image
end
img_std = img_std/mean(img_std);    %normalize
img_int = img_int/mean(img_int);    %ditto
img_score = img_std+img_int;    %create an image score
img_ptr(1) = find(img_score==min(img_score));  %We want the lowest score
img_ptr(2) = find(img_score==max(img_score));  %and the highest score

%prepare the stk for analysis
prompt_box('title','Find the nuclei','prompt1','Lets get interactive... ',...
    'prompt2','default Threshold = 0.2','position','center');
pause(0.25);
[thresh] = play_thresh(stk,img_ptr,0.05,zfilter);     %use the middle image as the representative image.
stk_bw = false(size(stk));      %preallocate
for i = 1:size(stk,3)     %make the stk Black & White
    stk_bw(:,:,i) = imfill(im2bw(stk(:,:,i),thresh),'holes'); %threshold should work fine at .1, change if you want.  Fill Holes.
end
%now filter out the objects that do not extend beyond 1 slice
if zfilter
    pixlst = regionprops(stk_bw,'PixelList');   %find the objects in the volume
    pixidxlst = regionprops(stk_bw,'PixelIdxList');     %get the pixel indexs for the objects
    pixlst = struct2cell(pixlst);    %convert to cell for simple useage
    for l = 1:size(pixlst,2)        %step through the array
        pixtmp(l) = size(unique(pixlst{1,l}(:,3)),1);   %we only care about whether the z is single slice or multi
    end
    rmvidx = find(pixtmp==1);   %find the single slice objects
    pixidxlst(rmvidx) = [];     %remove the single slice objects
    %now convert back to image for later calculations...admittedly I don't
    %need to do this, but to keep the rest of the function the same, I will
    %do this extra step.
    object_idx = cell2mat(struct2cell(pixidxlst)');    %grab the indexs of all of the pixels
    object_tmp = false(size(stk_bw));    %stack holder
    object_tmp(object_idx) = 1;     %create the image
    stk_bw = object_tmp;   %store
end
fill_pix = regionprops(~stk_bw,'FilledArea'); %find the number of pixels not occupied by the nucleus
fill_pix = sum(cell2mat(struct2cell(fill_pix)));    %pull the number out of data structure.
data(1).nucli_thresh = thresh;      %FYI store the threshold
dapi = regionprops(stk_bw,'FilledArea');    %dapi area
dapi = sum(cell2mat(struct2cell(dapi)));
data(1).nodapi = fill_pix*z;  %the no dapi area
data(1).dapi = dapi*z;    %the dapi area
% data(1).dapi2 = size(stk_bw(stk_bw==1),1);
% data(1).nodapi2 = size(stk_bw(stk_bw==0),1);
%now calculate the number of voxels, & remove the out of bound pixels as
%well
fill_pix = (fill_pix-obound)*z;          %if the pixels are isotropic, then fill_pix is the number of voxels.

%finally calculate the fill stat
if vert
    for j = 1:size(vertices,2)      %step through the number of channels opened.
        stat(1,j) = size(vertices{j},1)/fill_pix;   %calculate the puncta per fillable volumn stat for each channel.
    end
end

%output data
if vert
    data(1).stat = stat;
    data(1).filenames = filenames;
    data(1).pathnames = pathnames;
else
    data(1).filenames = filename;
    data(1).pathnames = pathname;
end
data(1).fill_vox = fill_pix;
data(1).obound = obound*z;
data(1).nucli_stk = stk_bw;
data(1).border_stk = stk_bw2;
data(1).total_vox = size(stk,1)*size(stk,2)*size(stk,3)*z;  %everything is modified by z, scaling, as a 1 x 1 x z

%localization for macs and linux
if ispc
    slash = '\';        %Windows directory marker
else
    slash = '/';        %Mac directory marker
end

%save the stacks automatically if selected.
stk_bw = im2uint16(stk_bw);  %make into a 16bit image
mkdir(pathname,'nuclei');
%matlabpool      %initiate processes
stk2tiff(stk_bw,filename{1,1},[pathname,'nuclei',slash]);
%matlabpool close %close processes

%--------------------------------------------------------------------------
function [thresh] = play_thresh(stk,img_ptr,thresh,zfilter,se)
%this function lets the user select a threshold interactively
%if inital thresh is not given, default to 0.1
if nargin<3
    thresh = 0.1;
    zfilter = 0;
    se = strel('disk',0);   %no dilation
elseif nargin<4
    zfilter = 0;
    se = strel('disk',0);   %no dilation
elseif nargin<5
    se = strel('disk',0);   %no dilation
end
ithresh = 0;        %initiate
warning('off')
%show original
h3 = figure('Name','Original High Contrast','NumberTitle','off');
imshow(stk(:,:,img_ptr(2)));
h1 = figure('Name','Original Low Contrast','NumberTitle','off');
imshow(stk(:,:,img_ptr(1)));
h = figure('Name','Processed Low Contrast','NumberTitle','off');     %create image
h2 = figure('Name','Processed High Contrast','NumberTitle','off');
%create short stacks so we don't have to process the whole stack everytime
if img_ptr(1)~=1 && img_ptr(1)~=size(stk,3)     %not the first or last slice
    stktmp1 = stk(:,:,img_ptr(1)-1:img_ptr(1)+1);   %short three stack
    idx1 = 2;   %slice index
elseif img_ptr(1)==1    %the first slice
    stktmp1 = stk(:,:,img_ptr(1):img_ptr(1)+2);   %short three stack
    idx1 = 1;
else    %last slice
    stktmp1 = stk(:,:,img_ptr(1)-2:img_ptr(1));   %short three stack
    idx1 = 3;
end
if img_ptr(2)~=1 && img_ptr(2)~=size(stk,3)     %not the first or last slice
    stktmp2 = stk(:,:,img_ptr(2)-1:img_ptr(2)+1);   %short three stack
    idx2 = 2;
elseif img_ptr(2)==1    %the first slice
    stktmp2 = stk(:,:,img_ptr(2):img_ptr(2)+2);   %short three stack
    idx2 = 1;
else    %last slice
    stktmp2 = stk(:,:,img_ptr(2)-2:img_ptr(2));   %short three stack
    idx2 = 3;
end
while ithresh==0
    for i = 1:3     %make the stk Black & White
        stk_bw1(:,:,i) = imfill(im2bw(stktmp1(:,:,i),thresh),'holes'); %threshold should work fine at .1, change if you want.  Fill Holes.
        stk_bw2(:,:,i) = imfill(im2bw(stktmp2(:,:,i),thresh),'holes');
    end
    %now filter out the objects that do not extend beyond 1 slice
    if zfilter
        %process image 1 first
        pixlst = regionprops(stk_bw1,'PixelList');   %find the objects in the volume
        pixidxlst = regionprops(stk_bw1,'PixelIdxList');     %get the pixel indexs for the objects
        pixlst = struct2cell(pixlst);    %convert to cell for simple useage
        for l = 1:size(pixlst,2)        %step through the array
            pixtmp(l) = size(unique(pixlst{1,l}(:,3)),1);   %we only care about whether the z is single slice or multi
        end
        rmvidx = find(pixtmp==1);   %find the single slice objects
        pixidxlst(rmvidx) = [];     %remove the single slice objects
        %now convert back to image for later calculations...admittedly I don't
        %need to do this, but to keep the rest of the function the same, I will
        %do this extra step.
        object_idx = cell2mat(struct2cell(pixidxlst)');    %grab the indexs of all of the pixels
        object_tmp = false(size(stk_bw1));    %stack holder
        object_tmp(object_idx) = 1;     %create the image
        stk_bw1 = object_tmp;   %store
        %process image 2 now
        pixlst2 = regionprops(stk_bw2,'PixelList');   %find the objects in the volume
        pixidxlst2 = regionprops(stk_bw2,'PixelIdxList');     %get the pixel indexs for the objects
        pixlst2 = struct2cell(pixlst2);    %convert to cell for simple useage
        for l = 1:size(pixlst2,2)        %step through the array
            pixtmp2(l) = size(unique(pixlst2{1,l}(:,3)),1);   %we only care about whether the z is single slice or multi
        end
        rmvidx2 = find(pixtmp2==1);   %find the single slice objects
        pixidxlst2(rmvidx2) = [];     %remove the single slice objects
        %now convert back to image for later calculations...admittedly I don't
        %need to do this, but to keep the rest of the function the same, I will
        %do this extra step.
        object_idx2 = cell2mat(struct2cell(pixidxlst2)');    %grab the indexs of all of the pixels
        object_tmp2 = false(size(stk_bw2));    %stack holder
        object_tmp2(object_idx2) = 1;     %create the image
        stk_bw2 = object_tmp2;   %store
    end
    clear rmvidx rmvidx2 pixtmp pixtmp2
    img = stk_bw1(:,:,idx1);     %pick image slice
    img2 = stk_bw2(:,:,idx2);     %pick image slice
    img_bw1 = imfill(imdilate(img,se),'holes');
    img_bw2 = imfill(imdilate(img2,se),'holes');
    figure(h2);     %select figure
    imshow(img_bw2);
    figure(h);     %select figure
    %hold
    imshow(img_bw1);
    ithresh = yes_no_box('title','Is this thresholded image OK?','caption1','Select yes is you want to move on.',...
        'caption2','Note: try an image that is background subtracted','position','west');
    pause(0.25);
    if ithresh==0
        thresh = str2double(response_box('title','Enter your new threshold','input',num2str(thresh),'position','center'));
        pause(0.25);
    else    %clean up and exit
        clear img img2
    end
end
close(h1);
close(h)
close(h2)
close(h3)
warning('on')

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [z,vert,border,zfilter] = parse(input)

z = 1;          %x,y,z = 1,1,1
vert = 0;       %Don't open a vertices file
border = 0;     %Don't calculate the border volume
zfilter = 0;    %no filtering

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'z'
                    z = input{1,i+1};
                case 'vert'
                    vert = input{1,i+1};
                case 'border'
                    border = input{1,i+1};
                case 'zfilter'
                    zfilter = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end